use Test::More;
use strict;
use warnings;
our ( $sub, $globref );

BEGIN {
	use_ok('Hades::Realm::OO');
	$sub     = sub { };
	$globref = \*globref;
}
subtest 'new' => sub {
	plan tests => 20;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	ok( $obj = Hades::Realm::OO->new(), q{$obj = Hades::Realm::OO->new()} );
	isa_ok( $obj, 'Hades::Realm::OO' );
	ok( $obj = Hades::Realm::OO->new(
			{   current_class => 'hypnos',
				is_role       => 1,
				meta          => {
					'nosoi' => {
						types      => { 'test' => 'test' },
						attributes => { 'test' => 'test' }
					}
				}
			}
		),
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => 1, meta => { 'nosoi' => { types => { 'test' => 'test' }, attributes => { 'test' => 'test' } } } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => [],
				is_role       => 1,
				meta          => {
					'nosoi' => {
						types      => { 'test' => 'test' },
						attributes => { 'test' => 'test' }
					}
				}
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => [], is_role => 1, meta => { 'nosoi' => { types => { 'test' => 'test' }, attributes => { 'test' => 'test' } } } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => \1,
				is_role       => 1,
				meta          => {
					'nosoi' => {
						types      => { 'test' => 'test' },
						attributes => { 'test' => 'test' }
					}
				}
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => \1, is_role => 1, meta => { 'nosoi' => { types => { 'test' => 'test' }, attributes => { 'test' => 'test' } } } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => 'hypnos',
				is_role       => [],
				meta          => {
					'nosoi' => {
						types      => { 'test' => 'test' },
						attributes => { 'test' => 'test' }
					}
				}
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => [], meta => { 'nosoi' => { types => { 'test' => 'test' }, attributes => { 'test' => 'test' } } } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => 'hypnos',
				is_role       => {},
				meta          => {
					'nosoi' => {
						types      => { 'test' => 'test' },
						attributes => { 'test' => 'test' }
					}
				}
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => {}, meta => { 'nosoi' => { types => { 'test' => 'test' }, attributes => { 'test' => 'test' } } } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => 'hypnos',
				is_role       => 1,
				meta          => {
					'nosoi' =>
					    { types => [], attributes => { 'test' => 'test' } }
				}
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => 1, meta => { 'nosoi' => { types => [], attributes => { 'test' => 'test' } } } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => 'hypnos',
				is_role       => 1,
				meta          => {
					'nosoi' => {
						types      => 'penthos',
						attributes => { 'test' => 'test' }
					}
				}
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => 1, meta => { 'nosoi' => { types => 'penthos', attributes => { 'test' => 'test' } } } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => 'hypnos',
				is_role       => 1,
				meta          => {
					'nosoi' => {
						types      => undef,
						attributes => { 'test' => 'test' }
					}
				}
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => 1, meta => { 'nosoi' => { types => undef, attributes => { 'test' => 'test' } } } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => 'hypnos',
				is_role       => 1,
				meta          => {
					'nosoi' =>
					    { types => { 'test' => 'test' }, attributes => [] }
				}
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => 1, meta => { 'nosoi' => { types => { 'test' => 'test' }, attributes => [] } } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => 'hypnos',
				is_role       => 1,
				meta          => {
					'nosoi' => {
						types      => { 'test' => 'test' },
						attributes => 'curae'
					}
				}
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => 1, meta => { 'nosoi' => { types => { 'test' => 'test' }, attributes => 'curae' } } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => 'hypnos',
				is_role       => 1,
				meta          => {
					'nosoi' => {
						types      => { 'test' => 'test' },
						attributes => undef
					}
				}
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => 1, meta => { 'nosoi' => { types => { 'test' => 'test' }, attributes => undef } } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => 'hypnos',
				is_role       => 1,
				meta          => { 'nosoi' => {} }
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => 1, meta => { 'nosoi' => {} } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => 'hypnos',
				is_role       => 1,
				meta          => { 'nosoi' => [] }
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => 1, meta => { 'nosoi' => [] } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => 'hypnos',
				is_role       => 1,
				meta          => { 'nosoi' => 'geras' }
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => 1, meta => { 'nosoi' => 'geras' } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   current_class => 'hypnos',
				is_role       => 1,
				meta          => { 'nosoi' => undef }
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => 1, meta => { 'nosoi' => undef } })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{ current_class => 'hypnos', is_role => 1, meta => [] } );
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => 1, meta => [] })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{ current_class => 'hypnos', is_role => 1, meta => 'curae' } );
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ current_class => 'hypnos', is_role => 1, meta => 'curae' })}
	);
};
subtest 'current_class' => sub {
	plan tests => 7;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'current_class' );
	is( $obj->current_class, undef, q{$obj->current_class} );
	is_deeply( $obj->current_class('aporia'),
		'aporia', q{$obj->current_class('aporia')} );
	eval { $obj->current_class( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->current_class([])} );
	eval { $obj->current_class( \1 ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->current_class(\1)} );
	is_deeply( $obj->current_class, 'aporia', q{$obj->current_class} );
};
subtest 'meta' => sub {
	plan tests => 17;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'meta' );
	is( $obj->meta, undef, q{$obj->meta} );
	is_deeply(
		$obj->meta(
			{   'aporia' => {
					types      => { 'test' => 'test' },
					attributes => { 'test' => 'test' }
				}
			}
		),
		{   'aporia' => {
				types      => { 'test' => 'test' },
				attributes => { 'test' => 'test' }
			}
		},
		q{$obj->meta({ 'aporia' => { types => { 'test' => 'test' }, attributes => { 'test' => 'test' } } })}
	);
	eval {
		$obj->meta(
			{   'aporia' =>
				    { types => [], attributes => { 'test' => 'test' } }
			}
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'aporia' => { types => [], attributes => { 'test' => 'test' } } })}
	);
	eval {
		$obj->meta(
			{   'aporia' =>
				    { types => 'curae', attributes => { 'test' => 'test' } }
			}
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'aporia' => { types => 'curae', attributes => { 'test' => 'test' } } })}
	);
	eval {
		$obj->meta(
			{   'aporia' =>
				    { types => undef, attributes => { 'test' => 'test' } }
			}
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'aporia' => { types => undef, attributes => { 'test' => 'test' } } })}
	);
	eval {
		$obj->meta(
			{   'aporia' =>
				    { types => { 'test' => 'test' }, attributes => [] }
			}
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'aporia' => { types => { 'test' => 'test' }, attributes => [] } })}
	);
	eval {
		$obj->meta(
			{   'aporia' =>
				    { types => { 'test' => 'test' }, attributes => 'curae' }
			}
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'aporia' => { types => { 'test' => 'test' }, attributes => 'curae' } })}
	);
	eval {
		$obj->meta(
			{   'aporia' =>
				    { types => { 'test' => 'test' }, attributes => undef }
			}
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'aporia' => { types => { 'test' => 'test' }, attributes => undef } })}
	);
	eval { $obj->meta( { 'aporia' => {} } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'aporia' => {} })}
	);
	eval { $obj->meta( { 'aporia' => [] } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'aporia' => [] })}
	);
	eval { $obj->meta( { 'aporia' => 'hypnos' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'aporia' => 'hypnos' })}
	);
	eval { $obj->meta( { 'aporia' => undef } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'aporia' => undef })}
	);
	eval { $obj->meta( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta([])} );
	eval { $obj->meta('penthos') };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta('penthos')} );
	is_deeply(
		$obj->meta,
		{   'aporia' => {
				types      => { 'test' => 'test' },
				attributes => { 'test' => 'test' }
			}
		},
		q{$obj->meta}
	);
};
subtest 'is_role' => sub {
	plan tests => 7;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'is_role' );
	is( $obj->is_role, undef, q{$obj->is_role} );
	is_deeply( $obj->is_role(1), 1, q{$obj->is_role(1)} );
	eval { $obj->is_role( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->is_role([])} );
	eval { $obj->is_role( {} ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->is_role({})} );
	is_deeply( $obj->is_role, 1, q{$obj->is_role} );
};
subtest 'clear_is_role' => sub {
	plan tests => 5;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'clear_is_role' );
	is_deeply( $obj->is_role(1), 1, q{$obj->is_role(1)} );
	ok( $obj->clear_is_role, q{$obj->clear_is_role} );
	is( $obj->is_role, undef, q{$obj->is_role} );
};
subtest 'module_generate' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'module_generate' );
	eval { $obj->module_generate( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->module_generate([])} );
	eval { $obj->module_generate('gaudia') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->module_generate('gaudia')}
	);
};
subtest 'build_class_inheritance' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_class_inheritance' );
};
subtest 'build_new' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_new' );
	eval { $obj->build_new( [], { 'test' => 'test' }, { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new([], { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->build_new(
			'phobos',
			{ 'test' => 'test' },
			{ 'test' => 'test' }
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new('phobos', { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval { $obj->build_new( bless( {}, 'Test' ), [], { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new(bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->build_new( bless( {}, 'Test' ), 'algea', { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new(bless({}, 'Test'), 'algea', { 'test' => 'test' })}
	);
	eval { $obj->build_new( bless( {}, 'Test' ), { 'test' => 'test' }, [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new(bless({}, 'Test'), { 'test' => 'test' }, [])}
	);
	eval {
		$obj->build_new( bless( {}, 'Test' ), { 'test' => 'test' }, 'limos' );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new(bless({}, 'Test'), { 'test' => 'test' }, 'limos')}
	);
};
subtest 'build_clearer' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_clearer' );
};
subtest 'build_predicate' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_predicate' );
};
subtest 'build_accessor_no_arguments' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_accessor_no_arguments' );
	eval {
		$obj->build_accessor_no_arguments( [], ['test'],
			{ 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_no_arguments([], ['test'], { 'test' => 'test' })}
	);
	eval {
		$obj->build_accessor_no_arguments( 'aporia', ['test'],
			{ 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_no_arguments('aporia', ['test'], { 'test' => 'test' })}
	);
	eval {
		$obj->build_accessor_no_arguments( bless( {}, 'Test' ),
			{}, { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_no_arguments(bless({}, 'Test'), {}, { 'test' => 'test' })}
	);
	eval {
		$obj->build_accessor_no_arguments( bless( {}, 'Test' ),
			'gaudia', { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_no_arguments(bless({}, 'Test'), 'gaudia', { 'test' => 'test' })}
	);
	eval {
		$obj->build_accessor_no_arguments( bless( {}, 'Test' ), ['test'], [] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_no_arguments(bless({}, 'Test'), ['test'], [])}
	);
	eval {
		$obj->build_accessor_no_arguments( bless( {}, 'Test' ),
			['test'], 'limos' );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_no_arguments(bless({}, 'Test'), ['test'], 'limos')}
	);
};
subtest 'build_accessor' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_accessor' );
	eval { $obj->build_accessor( [], 'aporia', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor([], 'aporia', { 'test' => 'test' })}
	);
	eval { $obj->build_accessor( 'aporia', 'aporia', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor('aporia', 'aporia', { 'test' => 'test' })}
	);
	eval {
		$obj->build_accessor( bless( {}, 'Test' ), [], { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor(bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->build_accessor( bless( {}, 'Test' ), \1, { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor(bless({}, 'Test'), \1, { 'test' => 'test' })}
	);
	eval { $obj->build_accessor( bless( {}, 'Test' ), 'aporia', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor(bless({}, 'Test'), 'aporia', [])}
	);
	eval { $obj->build_accessor( bless( {}, 'Test' ), 'aporia', 'geras' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor(bless({}, 'Test'), 'aporia', 'geras')}
	);
};
subtest 'build_modify' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_modify' );
	eval { $obj->build_modify( [], 'penthos', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_modify([], 'penthos', { 'test' => 'test' })}
	);
	eval { $obj->build_modify( 'nosoi', 'penthos', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_modify('nosoi', 'penthos', { 'test' => 'test' })}
	);
	eval {
		$obj->build_modify( bless( {}, 'Test' ), [], { 'test' => 'test' } );
	};
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_modify(bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->build_modify( bless( {}, 'Test' ), \1, { 'test' => 'test' } );
	};
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_modify(bless({}, 'Test'), \1, { 'test' => 'test' })}
	);
	eval { $obj->build_modify( bless( {}, 'Test' ), 'penthos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_modify(bless({}, 'Test'), 'penthos', [])}
	);
	eval { $obj->build_modify( bless( {}, 'Test' ), 'penthos', 'algea' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_modify(bless({}, 'Test'), 'penthos', 'algea')}
	);
};
subtest 'after_class' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'after_class' );
	eval { $obj->after_class( [], { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->after_class([], { 'test' => 'test' })}
	);
	eval { $obj->after_class( 'curae', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->after_class('curae', { 'test' => 'test' })}
	);
	eval { $obj->after_class( bless( {}, 'Test' ), [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->after_class(bless({}, 'Test'), [])}
	);
	eval { $obj->after_class( bless( {}, 'Test' ), 'hypnos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->after_class(bless({}, 'Test'), 'hypnos')}
	);
};
subtest 'unique_types' => sub {
	plan tests => 5;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'unique_types' );
	eval { $obj->unique_types( \1, { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->unique_types(\1, { 'test' => 'test' })}
	);
	eval { $obj->unique_types( 'hypnos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->unique_types('hypnos', [])}
	);
	eval { $obj->unique_types( 'hypnos', 'gaudia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->unique_types('hypnos', 'gaudia')}
	);
};
subtest 'build_as_class' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_as_class' );
	eval { $obj->build_as_class( [], { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_class([], { 'test' => 'test' })}
	);
	eval { $obj->build_as_class( 'phobos', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_class('phobos', { 'test' => 'test' })}
	);
	eval { $obj->build_as_class( bless( {}, 'Test' ), [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_class(bless({}, 'Test'), [])}
	);
	eval { $obj->build_as_class( bless( {}, 'Test' ), 'limos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_class(bless({}, 'Test'), 'limos')}
	);
};
subtest 'build_as_role' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_as_role' );
	eval { $obj->build_as_role( [], { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_role([], { 'test' => 'test' })}
	);
	eval { $obj->build_as_role( 'hypnos', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_role('hypnos', { 'test' => 'test' })}
	);
	eval { $obj->build_as_role( bless( {}, 'Test' ), [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_role(bless({}, 'Test'), [])}
	);
	eval { $obj->build_as_role( bless( {}, 'Test' ), 'hypnos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_role(bless({}, 'Test'), 'hypnos')}
	);
};
subtest 'build_has_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_has_keywords' );
	eval { $obj->build_has_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_has_keywords({})}
	);
	eval { $obj->build_has_keywords('curae') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_has_keywords('curae')}
	);
};
subtest 'build_has' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_has' );
	eval { $obj->build_has( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_has([])} );
	eval { $obj->build_has('geras') };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_has('geras')} );
};
subtest 'build_extends_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_extends_keywords' );
	eval { $obj->build_extends_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_extends_keywords({})}
	);
	eval { $obj->build_extends_keywords('limos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_extends_keywords('limos')}
	);
};
subtest 'build_extends' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_extends' );
	eval { $obj->build_extends( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_extends([])} );
	eval { $obj->build_extends('limos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_extends('limos')}
	);
};
subtest 'build_with_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_with_keywords' );
	eval { $obj->build_with_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_with_keywords({})}
	);
	eval { $obj->build_with_keywords('nosoi') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_with_keywords('nosoi')}
	);
};
subtest 'build_with' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_with' );
	eval { $obj->build_with( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_with([])} );
	eval { $obj->build_with('curae') };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_with('curae')} );
};
subtest 'build_requires_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_requires_keywords' );
	eval { $obj->build_requires_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_requires_keywords({})}
	);
	eval { $obj->build_requires_keywords('phobos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_requires_keywords('phobos')}
	);
};
subtest 'build_requires' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_requires' );
	eval { $obj->build_requires( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_requires([])} );
	eval { $obj->build_requires('limos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_requires('limos')}
	);
};
subtest 'build_before_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_before_keywords' );
	eval { $obj->build_before_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_before_keywords({})}
	);
	eval { $obj->build_before_keywords('aporia') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_before_keywords('aporia')}
	);
};
subtest 'build_before' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_before' );
	eval { $obj->build_before( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_before([])} );
	eval { $obj->build_before('limos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_before('limos')}
	);
};
subtest 'build_around_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_around_keywords' );
	eval { $obj->build_around_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_around_keywords({})}
	);
	eval { $obj->build_around_keywords('hypnos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_around_keywords('hypnos')}
	);
};
subtest 'build_around' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_around' );
	eval { $obj->build_around( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_around([])} );
	eval { $obj->build_around('thanatos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_around('thanatos')}
	);
};
subtest 'build_after_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_after_keywords' );
	eval { $obj->build_after_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_after_keywords({})}
	);
	eval { $obj->build_after_keywords('geras') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_after_keywords('geras')}
	);
};
subtest 'build_after' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_after' );
	eval { $obj->build_after( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_after([])} );
	eval { $obj->build_after('nosoi') };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_after('nosoi')} );
};
subtest 'build_accessor_coerce' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_accessor_coerce' );
	eval { $obj->build_accessor_coerce( [], 'curae' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_coerce([], 'curae')}
	);
	eval { $obj->build_accessor_coerce( \1, 'curae' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_coerce(\1, 'curae')}
	);
	eval { $obj->build_accessor_coerce( 'algea', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_coerce('algea', [])}
	);
	eval { $obj->build_accessor_coerce( 'algea', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_coerce('algea', \1)}
	);
};
subtest 'build_accessor_trigger' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_accessor_trigger' );
	eval { $obj->build_accessor_trigger( [], 'curae' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_trigger([], 'curae')}
	);
	eval { $obj->build_accessor_trigger( \1, 'curae' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_trigger(\1, 'curae')}
	);
	eval { $obj->build_accessor_trigger( 'algea', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_trigger('algea', [])}
	);
	eval { $obj->build_accessor_trigger( 'algea', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_trigger('algea', \1)}
	);
};
subtest 'build_accessor_default' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_accessor_default' );
	eval { $obj->build_accessor_default( [], 'geras' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_default([], 'geras')}
	);
	eval { $obj->build_accessor_default( \1, 'geras' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_default(\1, 'geras')}
	);
	eval { $obj->build_accessor_default( 'hypnos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_default('hypnos', [])}
	);
	eval { $obj->build_accessor_default( 'hypnos', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_default('hypnos', \1)}
	);
};
done_testing();
